# mysqlbackup v3.1
#
# Copyright 2009-2014 Kenneth Fribert
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#
# Script to make automatic backups of all your mysqldatabases
#
# Thanks to adnovea, patbaker82, silas, Don, Micke for inspiration and help, and 
#           jm45 and bramschats for testing
# PID file handling functions from "Subjectively" website
# There is no longer any user-modifyable section in this program, it's now in
# a seperate config file
# 

# Read config file
CONFIG="/etc/config/mysqlbackup.conf"
DAY_RET=`/sbin/getcfg mysqlbackup day_retention -f $CONFIG`
WEEK_RET=`/sbin/getcfg mysqlbackup week_retention -f $CONFIG`
MONTH_RET=`/sbin/getcfg mysqlbackup month_retention -f $CONFIG`
WEEKDAY_ROT=`/sbin/getcfg mysqlbackup day_rotate -f $CONFIG`
SHARE=`/sbin/getcfg mysqlbackup share -f $CONFIG`
FOLDER=`/sbin/getcfg mysqlbackup folder -f $CONFIG`
USER=`/sbin/getcfg mysqlbackup user -f $CONFIG`
PW=`/sbin/getcfg mysqlbackup pw -f $CONFIG`
LEVEL=`/sbin/getcfg mysqlbackup errorlvl -f $CONFIG`
 
# Standard commands used in this script
RM_C="/bin/rm"
SED_C="/bin/sed"
TAR_C="/bin/tar"
AWK_C="/bin/awk"
GET_C="/sbin/getcfg"
EC_C="/bin/echo"
LOG_C="/sbin/write_log"
MD_C="/bin/mkdir"
LS_C="/bin/ls"

# Internal variable setup
ARC=`date +%y%m%d`.tar.gz
DEST=
MONTH_DAY=`date +"%d"`
WEEK_DAY=`date +"%u"`
MYSQL_C=
MYSQLD_P=
MYSQLD_C=
MYSQLC_P=
MYSQLC_C=
ERROR=
DATABASES=

# Error and logging functions

function error ()
{ $EC_C -e "ERROR $1" ; if test "$LEVEL" -gt 0 ; then $LOG_C "MySQL Backup: ERROR $1" 1 ; fi ; exit 1 ; }

function warn ()
{ $EC_C -e "WARNING $1" ; if test "$LEVEL" -gt 1 ; then $LOG_C "MySQL Backup: WARNING $1" 2 ; fi ;  }

function info ()
{ $EC_C -e "MySQL Backup: $1" ; if test "$LEVEL" -gt 2 ; then $LOG_C "MySQL Backup: $1" 4 ; fi ; } 

# Functions for handling PID file

function pidfilename() {
  myfile=$(basename "$0" .sh)
  whoiam=$(whoami)
  mypidfile=/tmp/$myfile.pid
  [[ "$whoiam" == 'root' ]] && mypidfile=/var/run/$myfile.pid
  echo $mypidfile
}

function cleanup () {
  trap - INT TERM EXIT
  [[ -f "$mypidfile" ]] && rm "$mypidfile"
  exit
}

function isrunning() {
  pidfile="$1"
  [[ ! -f "$pidfile" ]] && return 1
  procpid=$(<"$pidfile")
  [[ -z "$procpid" ]] && return 1
  [[ ! $(ps -p $procpid | grep $(basename $0)) == "" ]] && value=0 || value=1
  return $value
}

function createpidfile() {
  mypid=$1
  pidfile=$2
  $(exec 2>&-; set -o noclobber; echo "$mypid" > "$pidfile") 
  [[ ! -f "$pidfile" ]] && exit #Lock file creation failed
  procpid=$(<"$pidfile")
  [[ $mypid -ne $procpid ]] && {
    isrunning "$pidfile" || {
      rm "$pidfile"
      $0 $@ &
    }
    {
    echo "MysqlBackup is already running, exiting"
    exit
    }
  }
}

# Start script
mypidfile=$(pidfilename)
createpidfile $$ "$mypidfile"
trap 'cleanup' INT TERM EXIT

# Checking if prerequisites are met
if [[ -z "$LEVEL" ]] ; then LEVEL="0" ; warnlater="Errorlevel not set in config, setting to 0 (nothing)" ; fi
$EC_C -e "\n"
info "MySQL Backup STARTED"

# Checking variables from config file
if [[ -n "$warnlater" ]] ; then warn "$warnlater" ; fi 
if [[ -z "$DAY_RET" ]] ; then DAY_RET="6" ; warn "days to keep backup not set in config, setting to 6" ; fi
if [[ -z "$WEEK_RET" ]] ; then WEEK_RET="5" ; warn "weeks to keep backup not set in config, setting to 5" ; fi
if [[ -z "$MONTH_RET" ]] ; then MONTH_RET="3" ; warn "months to keep backup not set in config, setting to 3" ; fi
if [[ -z "$WEEKDAY_ROT" ]] ; then WEEKDAY_ROT="0" ; warn "weekly rotate day not set in config, setting to sunday" ; fi
if [[ -z "$SHARE" ]] ; then SHARE="Backup" ; warn "share for storing Backup not set in config, setting to Backup" ; fi
if [[ -z "$USER" ]] ; then USER="User" ; warn "MySQL user for backup not set in config, setting to User" ; fi
if [[ -z "$PW" ]] ; then PW="Password" ; warn "MySQL password for backup not set in config, setting to Password" ; fi

# Check for backup share
BKUP_P=`$GET_C $SHARE path -f /etc/config/smb.conf`
if [ $? != 0 ] ; then error "the share $SHARE is not found, remember that the destination has to be a share" ; else info "Backup share found" ; fi

# Add subfolder to backup share
if [[ -z "$FOLDER" ]] ; then info "No subfolder given";
   else
   {
   BKUP_P=$BKUP_P"/"$FOLDER;
   # Check for subfolder under share
   $MD_C -p $BKUP_P ; if [ $? != 0 ] ; then error "the backup folder ($FOLDER) under the share could not be created on the share $SHARE" ; fi
   }
fi

# Check for backup folder on backup share
if ! [ -d $BKUP_P/mysql ] ; then info "mysql folder missing under $BKUP_P, it has been created" ; $MD_C $BKUP_P/mysql ; if [ $? != 0 ] ; then error "the folder mysql could not be created on the share $SHARE" ; fi ; fi

# Check for day retention folder on backup share
if ! [ -d $BKUP_P/mysql.daily ] ; then info "mysql.daily folder missing under the share $BKUP_P, it has been created" ; $MD_C $BKUP_P/mysql.daily ; if [ $? != 0 ] ; then error "the folder mysql.daily could not be created on the share $SHARE" ; fi ; fi

# Check for week retention folder on backup share
if ! [ -d $BKUP_P/mysql.weekly ] ; then info "mysql.weekly folder missing under the share $BKUP_P, it has been created" ; $MD_C $BKUP_P/mysql.weekly ; if [ $? != 0 ] ; then error "the folder mysql.weekly could not be created on the share $SHARE" ; fi ; fi

# Check for month retention folder on backup share
if ! [ -d $BKUP_P/mysql.monthly ] ; then info "mysql.monthly folder missing under the share $BKUP_P, it has been created" ; $MD_C $BKUP_P/mysql.monthly ; if [ $? != 0 ] ; then error "the folder mysql.monthly could not be created on the share $SHARE" ; fi ; fi

# Check for mysqldump command
for MYSQLD_P in /mnt/ext/opt/mysql /usr/local/mysql /mnt/HDA_ROOT/mysql /mnt/HDB_ROOT/mysql /mnt/HDC_ROOT/mysql /mnt/HDD_ROOT/mysql /mnt/HDE_ROOT/mysql /mnt/HDF_ROOT/mysql /mnt/HDG_ROOT/mysql /mnt/HDH_ROOT/mysql /share/MD0_DATA/.qpkg/Optware; do
  [ -f $MYSQLD_P/bin/mysqldump ] && MYSQLD_C="$MYSQLD_P/bin/mysqldump"
done
if [ -z $MYSQLD_C ] ; then error "mysqldump command not found."; else info "mysqldump command found" ; fi

# Check if mysqldump supports routines switch
if [[ "`$MYSQLD_C --routines`" = *"unknown"* ]]
 then
   MYSQLD_C=$MYSQLD_C" -eqQ --single-transaction"
 else
   MYSQLD_C=$MYSQLD_C" -eqQR --single-transaction"
fi

# Check for mysql command
for MYSQLC_P in /mnt/ext/opt/mysql /usr/local/mysql /mnt/HDA_ROOT/mysql /mnt/HDB_ROOT/mysql /mnt/HDC_ROOT/mysql /mnt/HDD_ROOT/mysql /mnt/HDE_ROOT/mysql /mnt/HDF_ROOT/mysql /mnt/HDG_ROOT/mysql /mnt/HDH_ROOT/mysql /share/MD0_DATA/.qpkg/Optware; do
  [ -f $MYSQLC_P/bin/mysql ] && MYSQLC_C="$MYSQLC_P/bin/mysql"
done
if [ -z $MYSQLC_C ] ; then error "mysql command not found.";  else info "mysql command found" ; fi

# Listing all the databases individually, and dumping them
DATABASES=`$MYSQLC_C -u $USER -p$PW -Bse 'show databases'`
if [ $? != 0 ] ; then error "cannot list databases, is password correct?" ; fi

# Delete old daily backups
info "Cleaning out old backups. Keeping the last $DAY_RET daily backups"
$RM_C -f `$LS_C -t $BKUP_P/mysql.daily/*  2> /dev/null | $AWK_C 'NR>'$DAY_RET`
if [ $? != 0 ] ; then error "ereasing old daily backups" ; fi

# Delete old weekly backups
info "Cleaning out old backups. Keeping the last $WEEK_RET week backups"
$RM_C -f `$LS_C -t $BKUP_P/mysql.weekly/*  2> /dev/null | $AWK_C 'NR>'$WEEK_RET`
if [ $? != 0 ] ; then error "ereasing old weekly backups" ; fi

# Delete old monthly backups
info "Cleaning out old backups. Keeping the last $MONTH_RET montly backups"
$RM_C -f `$LS_C -t $BKUP_P/mysql.monthly/*  2> /dev/null | $AWK_C 'NR>'$MONTH_RET`
if [ $? != 0 ] ; then error "ereasing old montly backups" ; fi

#info "Cleaning out yesterdays backup"
$RM_C -f $BKUP_P/mysql/*

#info "Backing up current databases to $BKUP_P/mysql"
while read LINE
do
  set $LINE
  $EC_C -e "Backing up database $LINE"
  $MYSQLD_C -u $USER -B $LINE --password=$PW > $BKUP_P/mysql/$LINE.sql
  if [ $? != 0 ]; then error "creating new backup when trying to access the database $LINE" ; ERROR=error ; fi
done<<<"$DATABASES"

if [[ -z $ERROR ]] ; then info "Backup Successfull" ; else error "Backup encountered errors, please investigate" ; fi
 
# Compress backup to an seleced archive

# On first month day do
if [ "$MONTH_DAY" == 1 ] ; then
  {
  DEST=mysql.monthly;
  info "Creating a monthly archive";
  }
else
  # On selected weekday do
  if [ $WEEK_DAY == $WEEKDAY_ROT ] ; then
    {
    DEST=mysql.weekly;
    info "Creating a weekly archive";
    }
  else
    # On any regular day do
    {
    DEST=mysql.daily;
    info "Creating a daily archive";
    }
  fi
fi

#info "Compressing backup to $BKUP_P/$DEST/$ARC"
cd $BKUP_P/mysql/ 
$TAR_C 2> /dev/null -czvf $BKUP_P/$DEST/$ARC * --remove-files
if [ $? != 0 ] ; then error "compressing backup" ; else info "Done compressing backup" ; fi

info "Backup script completed"
